
/*
Workfront Fusion Coder

@license Copyright 2021 Workfront. All right reserved.
 */

(function() {
  "use strict";
  var AST_TO_DOM, CREATE_FUNCTION, CREATE_NL, CREATE_PILL, Coder, DOM_TO_AST, DRAG_MOVE, ELEMENT_IN_FUNCTION, FIND_FUNCTION, FIND_KEYWORD, FIND_OPERATOR, FIND_VARIABLE, GET_CARET, GET_RANGE, GROUP_ICONS, HintPanel, INJECT_TEXT_NODE, INSTANCES, IS_NUMBER, IS_SPACER, LAST_ACTIVE, MD_ON_PILL, PillsPanel, SAMPLE, SET_CARET, TEXT_TO_IML_TYPE,
    extend = function(child, parent) { for (var key in parent) { if (hasProp.call(parent, key)) child[key] = parent[key]; } function ctor() { this.constructor = child; } ctor.prototype = parent.prototype; child.prototype = new ctor(); child.__super__ = parent.prototype; return child; },
    hasProp = {}.hasOwnProperty;

  if ('undefined' === typeof imt) {
    return;
  }

  INSTANCES = [];

  LAST_ACTIVE = null;

  MD_ON_PILL = false;

  imt.plugins.push({
    name: 'Coder',
    version: '1.16.3',
    author: 'Workfront'
  });

  SIMElement.prototype.coder = function(config) {
    if (this.data('coder') instanceof Coder) {
      return this.data('coder');
    }
    return new Coder(this, config);
  };

  DRAG_MOVE = null;

  CREATE_FUNCTION = function(pill) {
    var args, fce, i, j, ref;
    fce = [pill];
    if (pill.text() === '()(') {
      pill.text('(');
    }
    args = parseInt(pill.data('arguments'));
    if (args > 1) {
      for (i = j = 1, ref = args - 1; 1 <= ref ? j <= ref : j >= ref; i = 1 <= ref ? ++j : --j) {
        fce.push(sim.span('.i-coder-pill.i-coder-function-sep').attr('contenteditable', false).text(';'));
      }
    }
    fce.push(sim.span('.i-coder-pill.i-coder-function-end').attr('contenteditable', false).text(')'));
    return fce;
  };

  sim.ready(function() {
    var doc;
    doc = sim(document);
    doc.on('touchstart', ".i-coder-pills", function(event) {
      MD_ON_PILL = true;
      return true;
    });
    doc.on('mousedown', ".i-coder-pills", function(event) {
      var target;
      MD_ON_PILL = true;
      target = sim(event.target);
      if (target.closest('.i-panel-scroll') && !target.is('.i-coder-pill')) {
        return false;
      }
      return true;
    });
    doc.on('mouseup touchend', ".i-coder-pills", function(event) {
      var clone, fce, target;
      if ((LAST_ACTIVE != null) && MD_ON_PILL) {
        LAST_ACTIVE._root.focus();
        target = sim(event.target);
        if (target.hasClass('i-coder-pill')) {
          if (target.hasClass('i-coder-function')) {
            fce = CREATE_FUNCTION(target.clone(true).write('('));
            LAST_ACTIVE.insert(sim.array.apply(sim, fce));
          } else {
            clone = target.clone(true);
            if (clone.data('parent-type') === 'array' && /^\[(\d*|n)\]$/.test(clone.text())) {
              clone.text("[1]");
            }
            LAST_ACTIVE.insert(clone);
          }
        }
      }
      MD_ON_PILL = false;
      return true;
    });
    doc.on('drop touchmove', function(event) {
      MD_ON_PILL = false;
      return true;
    });
    doc.on('dragstart', ".i-coder-pill", function(event) {
      var fce, level, pill, ref, sibling, str;
      pill = this;
      if (pill.hasClass('i-coder-function')) {
        if (pill.parent().hasClass('i-coder-root')) {
          fce = [pill];
          level = 1;
          sibling = pill.next();
          while (sibling) {
            fce.push(sibling);
            sibling = sibling.next();
            if (sibling.hasClass('i-coder-function')) {
              level++;
            } else if (sibling.hasClass('i-coder-function-end')) {
              level--;
              if (level <= 0) {
                fce.push(sibling);
                break;
              }
            }
          }
          if (!sibling) {
            fce = [pill, sim.span('.i-coder-pill.i-coder-function-end')];
          }
          DRAG_MOVE = fce;
        } else {
          fce = CREATE_FUNCTION(pill);
        }
        str = IML.stringify(DOM_TO_AST(fce.slice(0)).ast);
      } else {
        if (pill.parent().hasClass('i-coder-root')) {
          DRAG_MOVE = [pill];
        }
        str = IML.stringify(DOM_TO_AST([pill], false).ast);
      }
      undelay(Coder._showHintTimer);
      if ((ref = Coder.hintPanel) != null) {
        ref.destroy();
      }
      event.effectAllowed = 'copy';
      event.dataTransfer.setData((browser.ie ? 'Text' : 'text/plain'), str);
      event.stopPropagation();
      return true;
    });
    return doc.on('dragend', '.i-coder-pill', function(event) {
      var coder, i, j, len, ref;
      MD_ON_PILL = false;
      if (DRAG_MOVE) {
        if (event.dataTransfer.dropEffect !== 'none') {
          if (typeof Surface !== "undefined" && Surface !== null) {
            if ((ref = Surface.Node.enlarged) != null) {
              ref.enlarged = false;
            }
          }
          for (j = 0, len = DRAG_MOVE.length; j < len; j++) {
            i = DRAG_MOVE[j];
            coder = i.closest('.i-coder');
            i.detach();
            if (coder != null) {
              coder._pull(coder.value);
            }
          }
        }
        DRAG_MOVE = null;
      } else {
        if (LAST_ACTIVE != null) {
          LAST_ACTIVE._root.focus();
        }
      }
      return true;
    });
  });

  GROUP_ICONS = {
    'best': '&#xF005;',
    'date': '&#xF073;',
    'math': '&#xF12B',
    'general': '&#xF013',
    'string': '&#xF031',
    'array': '&#xF00b',
    'fusion': imt.logo
  };

  IS_NUMBER = function(value) {
    return /^\-?\d+(\.\d+)?$/.test(value);
  };

  FIND_VARIABLE = function(variables, name, create) {
    var arrayIndexes, coder, elm, i, id, label, module, search, searchName;
    if (create == null) {
      create = true;
    }
    coder = this;
    arrayIndexes = [];
    searchName = name.replace(/\[(\d*)\]/g, function(a, b) {
      arrayIndexes.push(b ? parseInt(b) : 1);
      return "[]";
    });
    search = Coder.variables[searchName];
    if (!search) {
      search = variables[searchName];
    }
    if (search) {
      elm = search.clone(true);
      if (arrayIndexes.length) {
        i = 0;
        elm.data('prefix', elm.data('prefix').replace(/\[1\]/g, function(a, b) {
          return "[" + arrayIndexes[i++] + "]";
        }));
        if (arrayIndexes.length - 1 === i && /^\[(\d*|n)\]$/.test(elm.text())) {
          elm.text("[" + arrayIndexes[i] + "]");
        }
        elm.data('name', name);
      }
    } else if (create) {
      if ((typeof Inspector !== "undefined" && Inspector !== null ? Inspector.instance : void 0) && /^(\d+)\.(.*)$/.exec(name)) {
        id = parseInt(RegExp.$1);
        label = IML._unescape(RegExp.$2);
        module = Inspector.instance._surface.find(Surface.Module).filter(function(item) {
          return item.id === id;
        })[0];
        if (module) {
          return sim.span('.i-coder-pill.i-coder-variable', function() {
            this.attr('contenteditable', false);
            this.attr('draggable', true);
            this.attr('data-name', name);
            if (imt.config.showIDs) {
              this.attr('data-prefix', id + ". ");
            }
            this.text(label);
            this.css({
              color: module.theme.toHex(),
              backgroundColor: 'transparent',
              boxShadow: "inset 0px 0px 0px 1px " + (module.theme.toHex())
            });
            this.on('mouseenter', function() {
              var n, p;
              n = name;
              p = '';
              if (/^\d+\.(.*)(\[(?:\d*|n)\])$/.exec(name)) {
                n = RegExp.$2;
                p = RegExp.$1;
              }
              module.enlarged = true;
              return coder._showHint(this, {
                name: n,
                code: name,
                label: label,
                theme: module.theme,
                node: module,
                type: 'any',
                help: 'This is a custom variable.',
                prefix: p
              });
            });
            return this.on('mouseleave', function() {
              module.enlarged = false;
              return coder._hideHint();
            });
          });
        }
      }
      elm = sim.span('.i-coder-pill.i-coder-variable.i-coder-unknown', function() {
        this.attr('contenteditable', false);
        this.attr('draggable', true);
        this.attr('data-name', name);
        return this.text(name);
      });
    }
    return elm;
  };

  FIND_FUNCTION = function(name, create) {
    var elm, search;
    if (create == null) {
      create = true;
    }
    search = Coder.functions[name];
    if (search) {
      elm = search.clone(true);
      elm.text((elm.data('name')) + "(");
    } else if (create) {
      elm = sim.span('.i-coder-pill.i-coder-function.i-coder-unknown', function() {
        this.attr('contenteditable', false);
        this.attr('draggable', true);
        this.attr('data-name', name);
        return this.text(name + "(");
      });
    }
    return elm;
  };

  FIND_OPERATOR = function(name, create) {
    var elm, search;
    if (create == null) {
      create = true;
    }
    search = Coder.operators[name];
    if (search) {
      elm = search.clone(true);
    } else if (create) {
      elm = sim.span('.i-coder-pill.i-coder-operator.i-coder-unknown', function() {
        this.attr('contenteditable', false);
        this.attr('draggable', true);
        this.attr('data-name', name);
        return this.text(name);
      });
    }
    return elm;
  };

  FIND_KEYWORD = function(name, create) {
    var elm, search;
    if (create == null) {
      create = true;
    }
    search = Coder.keywords[name];
    if (search) {
      elm = search.clone(true);
    } else if (create) {
      elm = sim.span('.i-coder-pill.i-coder-keyword.i-coder-unknown', function() {
        this.attr('contenteditable', false);
        this.attr('draggable', true);
        this.attr('data-name', name);
        return this.text(name);
      });
    }
    return elm;
  };

  TEXT_TO_IML_TYPE = function(node) {
    if (IS_NUMBER(node.value)) {
      node.type = 'number';
      node.value = parseFloat(node.value);
    } else {
      node.type = 'string';
    }
    return node;
  };

  AST_TO_DOM = function(ast, parent, variables, incode) {
    var containsText, exp, index, render;
    if (incode == null) {
      incode = false;
    }
    exp = [];
    index = 0;
    containsText = false;
    render = (function(_this) {
      return function(ast) {
        var expr, i, ii, item, j, l, len, len1, len2, m, quotes, ref, ref1;
        quotes = false;
        if (incode && containsText && ast.length === 1 && (ast[0].value === '' || IS_NUMBER(ast[0].value))) {
          quotes = true;
          parent.write("\"");
        }
        for (j = 0, len = ast.length; j < len; j++) {
          item = ast[j];
          switch (item.type) {
            case 'variable':
              parent.append(FIND_VARIABLE.call(_this, variables, item.name));
              break;
            case 'function':
              parent.append(FIND_FUNCTION(item.name));
              ref = item["arguments"];
              for (i = l = 0, len1 = ref.length; l < len1; i = ++l) {
                expr = ref[i];
                if (i > 0) {
                  parent.span('.i-coder-pill.i-coder-function-sep', function() {
                    this.attr('contenteditable', false);
                    return this.text(';');
                  });
                }
                AST_TO_DOM.call(_this, expr, parent, variables, true);
              }
              parent.span('.i-coder-pill.i-coder-function-end', function() {
                this.attr('contenteditable', false);
                return this.text(')');
              });
              break;
            case 'operator':
              if (!(containsText && item.name === '+')) {
                parent.append(FIND_OPERATOR(item.name));
              }
              break;
            case 'number':
              parent.write(item.value.toString());
              break;
            case 'text':
            case 'string':
              ref1 = item.value.toString().split('\n');
              for (ii = m = 0, len2 = ref1.length; m < len2; ii = ++m) {
                i = ref1[ii];
                if (ii > 0) {
                  parent.append(CREATE_NL());
                }
                if (i.length > 0) {
                  parent.write(i);
                }
              }
              break;
            case 'keyword':
              parent.append(FIND_KEYWORD(item.name));
          }
        }
        if (quotes) {
          return parent.write("\"");
        }
      };
    })(this);
    while (ast[index]) {
      if (incode) {
        if (ast[index].type === 'operator' && ast[index].name !== '+') {
          render(exp);
          render([ast[index]]);
          exp.splice(0, exp.length);
          containsText = false;
          index++;
          continue;
        } else if (ast[index].type === 'string') {
          containsText = true;
        }
      }
      exp.push(ast[index]);
      index++;
    }
    if (exp.length) {
      render(exp);
    }
    return parent;
  };

  DOM_TO_AST = function(children, removeUnwantedOperators) {
    var elm, elms, errors, fce, finalize, index, layer, newlayer, prev, stack, text;
    if (removeUnwantedOperators == null) {
      removeUnwantedOperators = true;
    }
    stack = [];
    layer = IML.Array();
    errors = [];
    finalize = function(layer, incode) {
      var containsNonOperatedString, index, operation, ref, ref1, ref10, ref11, ref12, ref13, ref2, ref3, ref4, ref5, ref6, ref7, ref8, ref9, subindex;
      if (incode == null) {
        incode = false;
      }
      if (!layer.length) {
        return layer;
      }
      index = 0;
      while (index < layer.length) {
        if (layer[index].type === 'operator') {
          if (removeUnwantedOperators) {
            if (index === 0 && layer[index].name !== '!') {
              layer.shift();
              continue;
            } else if (index === layer.length - 1) {
              layer.pop();
              break;
            } else if (layer[index + 1].type === 'operator' && layer[index + 1].name !== '!') {
              layer.splice(index, 1);
              continue;
            }
          }
          if (!incode) {
            if (((ref = layer[index - 1]) != null ? ref.type : void 0) === 'text') {
              TEXT_TO_IML_TYPE(layer[index - 1]);
            }
            if (((ref1 = layer[index + 1]) != null ? ref1.type : void 0) === 'text') {
              TEXT_TO_IML_TYPE(layer[index + 1]);
            }
          }
        }
        index++;
      }
      if (incode && layer.length > 1) {
        containsNonOperatedString = false;
        if (layer[0].type === 'string' && layer[layer.length - 1].type === 'string') {
          if (layer[0].value.charAt(0) === '"' && layer[layer.length - 1].value.slice(-1) === '"') {
            layer[0].value = layer[0].value.slice(1);
            layer[layer.length - 1].value = layer[layer.length - 1].value.slice(0, -1);
            if (layer[0].value === '') {
              layer.shift();
            }
            if (layer[layer.length - 1].value === '') {
              layer.pop();
            }
          }
        }
        index = 0;
        while (index < layer.length) {
          if (layer[index].type === 'string' && ((ref2 = layer[index - 1]) != null ? ref2.type : void 0) !== 'operator' && ((ref3 = layer[index + 1]) != null ? ref3.type : void 0) !== 'operator') {
            containsNonOperatedString = true;
            break;
          }
          index++;
        }
        if (containsNonOperatedString) {
          index = 1;
          while (index < layer.length) {
            if (layer[index].type === 'operator') {
              subindex = index;
              operation = IML.Array();
              operation.push(layer[subindex - 1]);
              while (((ref4 = layer[subindex]) != null ? ref4.type : void 0) === 'operator') {
                operation.push(layer[subindex]);
                operation.push(layer[subindex + 1]);
                subindex += 2;
              }
              layer.splice(index - 1, operation.length, {
                type: 'function',
                name: '',
                "arguments": [operation]
              });
            }
            index++;
          }
        }
        index = 0;
        while (index < layer.length) {
          if (layer[index].type === 'string') {
            if ((ref5 = (ref6 = layer[index - 1]) != null ? ref6.type : void 0) === 'variable' || ref5 === 'function' || ref5 === 'keyword') {
              layer.splice(index++, 0, {
                type: 'operator',
                name: '+'
              });
            }
            if ((ref7 = (ref8 = layer[index + 1]) != null ? ref8.type : void 0) === 'variable' || ref7 === 'function' || ref7 === 'keyword') {
              layer.splice(index++ + 1, 0, {
                type: 'operator',
                name: '+'
              });
            }
          } else if ((ref9 = layer[index].type) === 'variable' || ref9 === 'function' || ref9 === 'keyword') {
            if ((ref10 = (ref11 = layer[index - 1]) != null ? ref11.type : void 0) === 'variable' || ref10 === 'function' || ref10 === 'keyword') {
              layer.splice(index++, 0, {
                type: 'operator',
                name: '+'
              });
            }
          } else if (layer[index].type === 'operator' && layer[index].name === '!') {
            if ((ref12 = (ref13 = layer[index - 1]) != null ? ref13.type : void 0) === 'variable' || ref12 === 'function' || ref12 === 'keyword') {
              layer.splice(index++, 0, {
                type: 'operator',
                name: '&'
              });
            }
          }
          index++;
        }
      }
      return layer;
    };
    prev = null;
    index = 0;
    elms = [];
    while (children.length) {
      elm = sim(children.shift());
      if (elm.nodeType === 3) {
        if ((prev != null ? prev.nodeType : void 0) === 3) {
          prev.text(prev.text() + elm.text());
          continue;
        }
        elm = sim.text(elm.text());
      } else if (elm.nodeType === 1 && elm.hasClass('i-coder-newline')) {
        if ((prev != null ? prev.nodeType : void 0) === 3) {
          prev.text(prev.text() + '\n');
          continue;
        }
        prev = sim.text('\n');
        elms.push(prev);
        continue;
      }
      prev = elm;
      elms.push(elm);
    }
    while (elms.length) {
      elm = elms.shift();
      if (elm.nodeType === 3) {
        text = elm.text().replace(/\u200B/g, '');
        if (text.length) {
          if (layer.fce == null) {
            layer.push({
              type: 'text',
              value: text
            });
          } else {
            if (text.trim().length > 0) {
              if (isNaN(text)) {
                if (/^\"([\s\S]*)\"$/.exec(text)) {
                  if (RegExp.$1 === '' || IS_NUMBER(RegExp.$1)) {
                    text = RegExp.$1;
                  }
                }
                layer.push({
                  type: 'string',
                  value: text
                });
              } else {
                layer.push({
                  type: 'number',
                  value: Number(text)
                });
              }
            }
          }
        }
      } else if (elm.hasClass('i-coder-variable')) {
        layer.push({
          type: 'variable',
          name: elm.attr('data-name')
        });
      } else if (elm.hasClass('i-coder-function')) {
        fce = {
          type: 'function',
          name: elm.attr('data-name'),
          "arguments": []
        };
        layer.push(fce);
        stack.push(layer);
        layer = IML.Array();
        layer.fce = fce;
        fce["arguments"].push(layer);
      } else if (elm.hasClass('i-coder-function-sep')) {
        if (!layer.fce) {
          errors.push(new Error("Argument separator outside function!"));
        } else {
          finalize(layer, true);
          newlayer = IML.Array();
          newlayer.fce = layer.fce;
          layer.fce["arguments"].push(newlayer);
          delete layer.fce;
          layer = newlayer;
        }
      } else if (elm.hasClass('i-coder-function-end')) {
        if (!layer.fce) {
          errors.push(new Error("Function end-bracket outside function!"));
        } else {
          finalize(layer, true);
          delete layer.fce;
          layer = stack.pop();
        }
      } else if (elm.hasClass('i-coder-operator')) {
        layer.push({
          type: 'operator',
          name: elm.attr('data-name')
        });
      } else if (elm.hasClass('i-coder-keyword')) {
        layer.push({
          type: 'keyword',
          name: elm.attr('data-name')
        });
      }
    }
    if (stack.length) {
      errors.push(new Error("Unclosed function found!"));
      while (stack.length) {
        layer = stack.pop();
      }
    }
    return {
      ast: finalize(layer),
      errors: errors
    };
  };

  SET_CARET = function(elm, offset) {
    var range, sel;
    offset = Math.max(offset, 0);
    if (elm.nodeType === 3) {
      offset = Math.min(elm.nodeValue.length, offset);
    }
    sel = window.getSelection();
    range = document.createRange();
    range.setStart(elm, offset);
    range.collapse(true);
    sel.removeAllRanges();
    return sel.addRange(range);
  };

  GET_CARET = function() {
    var ref, ref1, ref2, sel;
    sel = window.getSelection();
    if (sel.focusNode == null) {
      return {
        anchor: null,
        offset: 0
      };
    }
    if (sel.focusNode.nodeType !== 3) {
      if (sel.focusNode.nodeType === 1) {
        if (((ref = sel.focusNode.childNodes[sel.focusOffset]) != null ? ref.nodeType : void 0) === 3) {
          return {
            anchor: sel.focusNode.childNodes[sel.focusOffset],
            offset: 0
          };
        } else if (((ref1 = sel.focusNode.childNodes[sel.focusOffset]) != null ? (ref2 = ref1.previousSibling) != null ? ref2.nodeType : void 0 : void 0) === 3) {
          return {
            anchor: sel.focusNode.childNodes[sel.focusOffset].previousSibling,
            offset: 0
          };
        } else {
          return {
            anchor: null,
            offset: 0
          };
        }
      } else {
        return {
          anchor: null,
          offset: 0
        };
      }
    } else {
      return {
        anchor: sel.focusNode,
        offset: sel.focusOffset
      };
    }
  };

  GET_RANGE = function(root) {
    var elm, end, endOffset, out, range, sel, start, startOffset;
    root = root.__dom;
    if (!root.childNodes.length) {
      return [];
    }
    sel = window.getSelection();
    if (sel.isCollapsed || !sel.rangeCount) {
      return [];
    }
    range = sel.getRangeAt(0);
    start = range.startContainer;
    startOffset = range.startOffset;
    end = range.endContainer;
    endOffset = range.endOffset;
    if (start === root) {
      start = root.childNodes[0];
      if (start.nodeType === 3) {
        startOffset = 0;
      }
    }
    if (end === root) {
      end = root.childNodes[root.childNodes.length - 1];
      if (end.nodeType === 3) {
        endOffset = end.nodeValue.length;
      }
    }
    if (!sim(root).contains(start || !sim(root).contains(end))) {
      return [];
    }
    if (start === end || (end == null)) {
      if (start.nodeType === 3) {
        return [document.createTextNode(start.nodeValue.substring(startOffset, endOffset))];
      } else {
        return [start];
      }
    }
    while (start.parentNode !== root) {
      start = start.parentNode;
    }
    while ((end != null) && end.parentNode !== root) {
      end = end.parentNode;
    }
    if (end == null) {
      if (start.nodeType === 3) {
        return [document.createTextNode(start.nodeValue.substring(startOffset, endOffset))];
      } else {
        return [start];
      }
    }
    out = [];
    elm = start.nextSibling;
    while (elm !== end) {
      out.push(elm);
      elm = elm.nextSibling;
    }
    if (start.nodeType === 3) {
      out.unshift(document.createTextNode(start.nodeValue.substr(startOffset)));
    } else {
      out.unshift(start);
    }
    if (end.nodeType === 3) {
      out.push(document.createTextNode(end.nodeValue.substr(0, endOffset)));
    } else {
      out.push(end);
    }
    return out;
  };

  ELEMENT_IN_FUNCTION = function(elm) {
    elm = sim(elm);
    return elm.prevAll('.i-coder-function').length > elm.prevAll('.i-coder-function-end').length;
  };

  INJECT_TEXT_NODE = function(text, offset, elm, caret) {
    var next, space;
    if (caret == null) {
      caret = true;
    }
    if (offset === text.nodeValue.length) {
      if (text.nextSibling != null) {
        if (caret) {
          if (text.nextSibling.nodeType === 3) {
            SET_CARET(text.nextSibling, (IS_SPACER(text.nextSibling) ? 1 : 0));
          } else {
            space = document.createTextNode("\u200B");
            text.parentNode.insertBefore(space, text.nextSibling);
            SET_CARET(space, 1);
          }
        }
        elm.insertBefore(text.nextSibling);
      } else {
        elm.appendTo(text.parentNode);
        space = document.createTextNode("\u200B");
        text.parentNode.appendChild(space);
        SET_CARET(space, 1);
      }
    } else if (offset === 0) {
      elm.insertBefore(text);
      if (caret) {
        SET_CARET(text, (IS_SPACER(text) ? 1 : 0));
      }
    } else {
      next = text.splitText(offset);
      elm.insertBefore(next);
      if (caret) {
        SET_CARET(next, 0);
      }
    }
    return null;
  };

  CREATE_PILL = function(klass, text) {
    if (text == null) {
      text = '';
    }
    return sim.span('.i-coder-pill', function() {
      this.attr('contenteditable', false);
      this.attr('draggable', true);
      this.addClass(klass);
      return this.text(text);
    });
  };

  CREATE_NL = function() {
    return sim.span('.i-coder-newline', function() {
      this.attr('contenteditable', false);
      return this.text('\n');
    });
  };

  IS_SPACER = function(anchor) {
    return anchor.nodeValue.indexOf('\u200B') !== -1;
  };

  SAMPLE = function(value) {
    if (value == null) {
      return '';
    }
    switch (typeof value) {
      case 'string':
        if (/^IMTBuffer\((\d+), ([^,]*), ([^,]*)\): ([\s\S]*)$/.exec(value)) {
          return RegExp.$4.replace(/(.{2})/g, '$1 ');
        }
        if (/^IMTString\((\d+)\): ([\s\S]*)$/.exec(value)) {
          return RegExp.$2;
        }
        return value;
      case 'object':
        if (Array.isArray(value)) {
          return "[" + (value.map(function(item) {
            return SAMPLE(item);
          }).join(', ')) + "]";
        }
        if (value instanceof Date) {
          return format(value, 'datetime', {
            timezone: imt.user.timezone
          });
        }
        return '{collection}';
      default:
        return String(value);
    }
  };


  /*
  Class
  
  @static pills {SIMElement} Root of the pills menu.
  @static functions {Object} Collection of system functions.
  @static variables {Object} Collection of system variables.
  @static operators {Object} Collection of system operators.
   */

  Coder = (function(superClass) {
    extend(Coder, superClass);

    Coder.pills = null;

    Coder.functions = {};

    Coder.variables = {};

    Coder.operators = {};

    Coder.keywords = {};

    Coder.prototype.errors = null;

    Coder.prototype.variables = null;

    Coder.prototype._cursor = null;

    Coder.prototype._preview = null;

    Coder.prototype.__preview = false;

    Coder.property('enabled', {
      get: function() {
        return !this.prop('disabled');
      },
      set: function(value) {
        this._root.attr('contenteditable', value);
        this.prop('disabled', !value);
        if (!value) {
          return this._hidePills();
        }
      }
    });

    Coder.property('preview', {
      get: function() {
        return this.__preview;
      },
      set: function(bool) {
        var ast, err, errors, j, len, ref;
        if (bool === this.__preview) {
          return;
        }
        this.__preview = bool;
        if (bool) {
          this._hidePills();
          ref = DOM_TO_AST(this._root.contents()), ast = ref.ast, errors = ref.errors;
          if (errors.length) {
            for (j = 0, len = errors.length; j < len; j++) {
              err = errors[j];
              console.warn(err.message);
            }
            return this.__preview = false;
          } else {
            this._root.hide();
            this._preview.text(IML.execute(ast, this.config.samples));
            return this._preview.show();
          }
        } else {
          this._root.show();
          return this._preview.empty().hide();
        }
      }
    });

    Coder.property('value', {
      get: function() {
        var ast, errors, ref, text;
        ref = DOM_TO_AST(this._root.contents()), ast = ref.ast, errors = ref.errors;
        text = IML.stringify(ast);
        this.errors = errors.concat(IML.errors);
        return text;
      },
      set: function(value) {
        var ast, err, j, len, ref;
        if (this.value === (value || '')) {
          return;
        }
        this._root.contents().detach();
        ast = IML.parse(value);
        AST_TO_DOM.call(this, ast, this._root, this.variables);
        this._pull(IML.stringify(ast));
        ref = IML.errors;
        for (j = 0, len = ref.length; j < len; j++) {
          err = ref[j];
          console.warn(err);
        }
        return this._normalize();
      }
    });


    /*
    	config:
    	- `type` - Data type of coder's value
     */

    function Coder(input, config) {
      var createPills, g, group, groups, i, j, k, l, len, len1, len2, m, ref, ref1, ref2, ref3, ref4, ref5, self, t, v;
      if (config == null) {
        config = {};
      }
      Coder.__super__.constructor.call(this, 'div', '.i-coder');
      this.data('coder', this);
      self = this;
      this.variables = {};
      this.config = config;
      this._input = input;
      this.insertBefore(input);
      this.append(input);
      if (input.css('display') === 'none') {
        this.hide();
      }
      input.hide();
      this.errors = [];
      this._root = this.div('.i-coder-root.form-control', function() {
        this.attr('contenteditable', true);
        this.attr('spellcheck', false);
        return this.attr('tabindex', 0);
      });
      if (!Coder.pills) {
        groups = {
          general: null,
          math: null,
          string: null,
          date: null,
          array: null,
          fusion: null
        };
        ref = ['functions', 'variables', 'operators', 'keywords'];
        for (i = j = 0, len = ref.length; j < len; i = ++j) {
          t = ref[i];
          ref1 = [IML.FUNCTIONS, IML.VARIABLES, IML.OPERATORS, IML.KEYWORDS][i];
          for (k in ref1) {
            v = ref1[k];
            if (!(v.hidden !== true)) {
              continue;
            }
            v.name = k;
            ref2 = [].concat(v.group);
            for (l = 0, len1 = ref2.length; l < len1; l++) {
              g = ref2[l];
              if (groups[g] == null) {
                groups[g] = {
                  variables: [],
                  functions: [],
                  operators: [],
                  keywords: []
                };
              }
              groups[g][t].push(v);
            }
          }
        }
        Coder.pillsHeader = sim.ul('.nav.nav-tabs', function() {
          var name, results;
          this.li('.nav-item').a('.nav-link.active', function() {
            this.attr('href', '#pills-best');
            this.attr('data-toggle', 'tab');
            this.attr('data-history', 'no');
            this.html(GROUP_ICONS.best);
            return this.on('shown.bs.tab', function() {
              var ref3;
              return (ref3 = Coder.pillsPanel) != null ? ref3._resize() : void 0;
            });
          });
          results = [];
          for (name in groups) {
            if (groups[name] != null) {
              results.push(this.li('.nav-item.').a('.nav-link', function() {
                var ref3;
                this.attr('href', "#pills-" + name);
                this.attr('data-toggle', 'tab');
                this.attr('data-history', 'no');
                this.attr('title', imt.l('iml', name + ".label"));
                if (/^(https:)?\/\//.test(GROUP_ICONS[name])) {
                  this.img().attr('src', GROUP_ICONS[name]).css({
                    width: 14,
                    height: 14,
                    transform: 'translate(0, -1px)'
                  });
                } else {
                  this.html((ref3 = GROUP_ICONS[name]) != null ? ref3 : name);
                }
                return this.on('shown.bs.tab', function() {
                  var ref4;
                  return (ref4 = Coder.pillsPanel) != null ? ref4._resize() : void 0;
                });
              }));
            }
          }
          return results;
        });
        Coder.pills = sim.div('.tab-content', function() {
          var group, name, results;
          this.div('#pills-best.tab-pane.active', function() {});
          results = [];
          for (name in groups) {
            group = groups[name];
            if (group != null) {
              results.push(this.div("#pills-" + name + ".tab-pane", function() {
                var len2, m, ref3, variable;
                this.h4().text(imt.l('iml', name + ".label"));
                if (group.variables.length) {
                  this.h5().text(imt.l('coder', "pills.variables"));
                  ref3 = group.variables;
                  for (m = 0, len2 = ref3.length; m < len2; m++) {
                    variable = ref3[m];
                    this.div('.i-coder-pill-row', function() {
                      this.div().span('.i-coder-pill.i-coder-variable', function() {
                        Coder.variables[variable.name] = this;
                        this.attr('contenteditable', false);
                        this.attr('draggable', true);
                        this.attr('data-name', variable.name);
                        this.attr('data-type', variable.type);
                        this.text(variable.name);
                        return (function(_this) {
                          return function(name, variable) {
                            _this.on('mouseenter', function() {
                              return self._showHint(this, {
                                name: variable.name,
                                help: imt.l('iml', name + ".variables." + variable.name + ".help"),
                                type: variable.type
                              });
                            });
                            return _this.on('mouseleave', function() {
                              return self._hideHint();
                            });
                          };
                        })(this)(name, variable);
                      });
                      return this.div('.i-coder-sample').text(variable.sample);
                    });
                  }
                }
                if (group.functions.length) {
                  this.h5().text(imt.l('coder', "pills.functions"));
                  this.div('.i-coder-pill-group', function() {
                    var fce, len3, o, ref4, results1;
                    ref4 = group.functions;
                    results1 = [];
                    for (o = 0, len3 = ref4.length; o < len3; o++) {
                      fce = ref4[o];
                      results1.push(this.span('.i-coder-pill.i-coder-function', function() {
                        Coder.functions[fce.name] = this;
                        this.attr('contenteditable', false);
                        this.attr('draggable', true);
                        this.attr('data-name', fce.name);
                        this.attr('data-type', fce.type);
                        this.attr('data-arguments', fce.value.length);
                        this.text(fce.name || '()');
                        return (function(_this) {
                          return function(name, fce) {
                            _this.on('mouseenter', function() {
                              return self._showHint(this, {
                                name: imt.l('iml', name + ".functions." + fce.name + ".syntax"),
                                help: imt.l('iml', name + ".functions." + fce.name + ".help"),
                                example: imt.l('iml', name + ".functions." + fce.name + ".example", {
                                  optional: true
                                }),
                                type: fce.type
                              });
                            });
                            return _this.on('mouseleave', function() {
                              return self._hideHint();
                            });
                          };
                        })(this)(name, fce);
                      }));
                    }
                    return results1;
                  });
                }
                if (group.operators.length) {
                  this.h5().text(imt.l('coder', "pills.operators"));
                  this.div('.i-coder-pill-group', function() {
                    var len3, o, op, ref4, results1;
                    ref4 = group.operators;
                    results1 = [];
                    for (o = 0, len3 = ref4.length; o < len3; o++) {
                      op = ref4[o];
                      results1.push(this.span('.i-coder-pill.i-coder-operator', function() {
                        var ref5;
                        Coder.operators[op.name] = this;
                        this.attr('contenteditable', false);
                        this.attr('draggable', true);
                        this.attr('data-name', op.name);
                        this.text((ref5 = op.label) != null ? ref5 : op.name);
                        return (function(_this) {
                          return function(name, op) {
                            _this.on('mouseenter', function() {
                              return self._showHint(this, {
                                name: op.name,
                                help: imt.l('iml', name + ".operators." + op.name + ".help")
                              });
                            });
                            return _this.on('mouseleave', function() {
                              return self._hideHint();
                            });
                          };
                        })(this)(name, op);
                      }));
                    }
                    return results1;
                  });
                }
                if (group.keywords.length) {
                  this.h5().text(imt.l('coder', "pills.keywords"));
                  return this.div('.i-coder-pill-group', function() {
                    var kw, len3, o, ref4, results1;
                    ref4 = group.keywords;
                    results1 = [];
                    for (o = 0, len3 = ref4.length; o < len3; o++) {
                      kw = ref4[o];
                      results1.push(this.span('.i-coder-pill.i-coder-keyword', function() {
                        Coder.keywords[kw.name] = this;
                        this.attr('contenteditable', false);
                        this.attr('draggable', true);
                        this.attr('data-name', kw.name);
                        this.text(kw.name);
                        return (function(_this) {
                          return function(name, kw) {
                            _this.on('mouseenter', function() {
                              return self._showHint(this, {
                                name: imt.l('iml', name + ".keywords." + kw.name + ".label"),
                                help: imt.l('iml', name + ".keywords." + kw.name + ".help")
                              });
                            });
                            return _this.on('mouseleave', function() {
                              return self._hideHint();
                            });
                          };
                        })(this)(name, kw);
                      }));
                    }
                    return results1;
                  });
                }
              }));
            }
          }
          return results;
        });
      }
      createPills = function(group, nested, level) {
        var index, len2, m, pill, ref3;
        if (level == null) {
          level = 0;
        }
        ref3 = group.pills;
        for (index = m = 0, len2 = ref3.length; m < len2; index = ++m) {
          pill = ref3[index];
          if (self.variables[pill.code]) {
            continue;
          }
          sim.span('.i-coder-pill.i-coder-variable', function() {
            var ref4;
            self.variables[pill.code] = this;
            this.attr('contenteditable', false);
            this.attr('draggable', true);
            this.attr('data-name', pill.code);
            this.attr('data-type', pill.type);
            this.attr('data-prefix', (imt.config.showIDs && pill.node ? pill.node.id + '. ' : '') + pill.prefix);
            this.attr('data-level', level);
            this.text(pill.name);
            if (nested) {
              this.attr('data-nested', true);
              this.attr('data-parent-type', group.type);
              this.attr('data-nth', index + 1);
            }
            if (pill.theme) {
              this.css({
                color: pill.theme.determineForegroundColor(),
                backgroundColor: pill.theme.toHex()
              });
            }
            (function(_this) {
              return (function(pill) {
                _this.on('mouseenter', function() {
                  var ref4;
                  if ((ref4 = pill.node) != null) {
                    ref4.enlarged = true;
                  }
                  return self._showHint(this, pill);
                });
                return _this.on('mouseleave', function() {
                  var ref4;
                  if ((ref4 = pill.node) != null) {
                    ref4.enlarged = false;
                  }
                  return self._hideHint();
                });
              });
            })(this)(pill);
            if (/^(\d+)\.(.*)$/.test(pill.code)) {
              this.data('sample', sim.div('.i-coder-sample').text(SAMPLE(IML._mapVariable(self.config.samples, pill.code))));
            }
            if ((ref4 = pill.pills) != null ? ref4.length : void 0) {
              this.attr('data-has-children', true);
              return createPills(pill, true, level + 1);
            }
          });
        }
        return null;
      };
      ref4 = (ref3 = config != null ? config.pills : void 0) != null ? ref3 : [];
      for (m = 0, len2 = ref4.length; m < len2; m++) {
        group = ref4[m];
        if (((ref5 = group.pills) != null ? ref5.length : void 0) > 0) {
          createPills(group);
        }
      }
      this._preview = sim.div('.i-coder-preview.form-control');
      this._preview.hide();
      this.append(this._preview);
      this.value = input.val();
      this.addClass(input.attr('class'));
      this.removeClass('form-control');
      this.enabled = input.enabled;
      this.on('keydown', (function(_this) {
        return function(event) {
          var anchor, offset, ref6, ref7, ref8;
          ref6 = GET_CARET(), anchor = ref6.anchor, offset = ref6.offset;
          if (!anchor) {
            return;
          }
          if (event.which === 8 && window.getSelection().isCollapsed) {
            if ((IS_SPACER(anchor) || offset === 0) && ((ref7 = anchor.previousSibling) != null ? ref7.nodeType : void 0) === 1) {
              anchor.parentNode.removeChild(anchor.previousSibling);
              event.preventDefault();
              _this.emit('input');
            } else if (anchor.nodeType === 3 && offset === 1 && anchor.nodeValue.length === 1) {
              anchor.nodeValue = "\u200B";
              event.preventDefault();
              SET_CARET(anchor, 1);
              _this.emit('input');
            }
          } else if (event.which === 46 && window.getSelection().isCollapsed) {
            if ((IS_SPACER(anchor) || offset === anchor.nodeValue.length) && ((ref8 = anchor.nextSibling) != null ? ref8.nodeType : void 0) === 1) {
              anchor.parentNode.removeChild(anchor.nextSibling);
              event.preventDefault();
              _this.emit('input');
            } else if (anchor.nodeType === 3 && offset === 0 && anchor.nodeValue.length === 1) {
              anchor.nodeValue = "\u200B";
              event.preventDefault();
              SET_CARET(anchor, 1);
              _this.emit('input');
            }
          } else if (IS_SPACER(anchor)) {
            switch (event.which) {
              case 39:
                if (offset === 0) {
                  SET_CARET(anchor, 1);
                }
                break;
              case 37:
                if (offset === 1) {
                  SET_CARET(anchor, 0);
                }
            }
          }
          return true;
        };
      })(this));
      this.on('keypress', (function(_this) {
        return function(event) {
          var anchor, fce, name, offset, ref10, ref6, ref7, ref8, ref9;
          switch (event.which) {
            case 59:
              ref6 = GET_CARET(), anchor = ref6.anchor, offset = ref6.offset;
              if (!anchor) {
                return;
              }
              if (ELEMENT_IN_FUNCTION(anchor)) {
                INJECT_TEXT_NODE(anchor, offset, CREATE_PILL('i-coder-function-sep', ';'));
                event.preventDefault();
                _this.emit('input');
              }

              /*when 43, 45, 42, 47 # + - * /
              						{anchor, offset} = GET_CARET()
              						unless anchor then return
              						
              						replace = false
              						lselector = '.i-coder-pill:not(.i-coder-function):not(.i-coder-function-sep)'
              						rselector = '.i-coder-pill:not(.i-coder-function-end):not(.i-coder-function-sep)'
              						
              						if anchor.nodeValue.indexOf('\u200B') isnt -1
              							 * we're on spacer
              							if sim(anchor.previousSibling)?.is(lselector) or sim(anchor.nextSibling)?.is(rselector)
              								replace = true
              						
              						else
              							 * we're on normal text node, check for edges
              							if (offset is 0 and sim(anchor.previousSibling)?.is(lselector)) or (offset is anchor.nodeValue.length and sim(anchor.nextSibling)?.is(rselector))
              								replace = true
              						
              						if replace and not sim(anchor.previousSibling)?.hasClass('i-coder-operator') and not sim(anchor.nextSibling)?.hasClass('i-coder-operator')
              							 * replace + with code
              							INJECT_TEXT_NODE anchor, offset, FIND_OPERATOR String.fromCharCode event.which
              	
              							event.preventDefault()
              							@emit 'input'
               */
              break;
            case 40:
              ref7 = GET_CARET(), anchor = ref7.anchor, offset = ref7.offset;
              if (!anchor) {
                return;
              }
              name = anchor.nodeValue.substr(0, offset);
              name = (ref8 = name.match(/[a-zA-Z][a-zA-Z0-9]*$/)) != null ? ref8[0] : void 0;
              fce = FIND_FUNCTION(name, false);
              if (fce) {
                anchor.nodeValue = "" + (anchor.nodeValue.substr(0, offset - name.length)) + (anchor.nodeValue.substr(offset));
                offset -= name.length;
                INJECT_TEXT_NODE(anchor, offset, fce);
                event.preventDefault();
                _this.emit('input');
              }
              break;
            case 41:
              ref9 = GET_CARET(), anchor = ref9.anchor, offset = ref9.offset;
              if (!anchor) {
                return;
              }
              if (ELEMENT_IN_FUNCTION(anchor)) {
                INJECT_TEXT_NODE(anchor, offset, CREATE_PILL('i-coder-function-end', ')'));
                event.preventDefault();
                _this.emit('input');
              }
              break;
            case 13:
              ref10 = GET_CARET(), anchor = ref10.anchor, offset = ref10.offset;
              if (!anchor) {
                return;
              }
              INJECT_TEXT_NODE(anchor, offset, CREATE_NL());
              event.preventDefault();
              _this.emit('input');
          }
          return true;
        };
      })(this));
      if (browser.ie) {
        this.on('keypress paste drop', (function(_this) {
          return function(event) {
            setImmediate(function() {
              return _this.emit('input');
            });
            return true;
          };
        })(this));
      }
      this.on('input', (function(_this) {
        return function(event) {
          var ref6;
          _this._normalize();
          _this._pull(_this.value);
          if ((ref6 = Coder.pillsPanel) != null) {
            ref6._resize();
          }
          return true;
        };
      })(this));
      this.on('copy', (function(_this) {
        return function(event) {
          var ast, range;
          event.preventDefault();
          event.stopImmediatePropagation();
          range = GET_RANGE(_this._root);
          if (!range.length) {
            return;
          }
          ast = DOM_TO_AST(range).ast;
          if (window.clipboardData) {
            window.clipboardData.setData('Text', IML.stringify(ast));
          } else {
            event.clipboardData.setData('text/plain', IML.stringify(ast));
          }
          return true;
        };
      })(this));
      this._root.on('focusin', (function(_this) {
        return function(event) {
          if (!_this.enabled) {
            _this._root.blur();
            return false;
          }
          LAST_ACTIVE = _this;
          _this._showPills();
          return true;
        };
      })(this));
      this._root.on('focusout', (function(_this) {
        return function(event) {
          if (!MD_ON_PILL) {
            _this._normalize();
            _this._pull(_this.value);
            _this._hidePills();
          }
          return true;
        };
      })(this));
    }

    Coder.prototype._showPills = function() {
      var best, panel, pillsFound, self;
      if (Coder.pillsPanel != null) {
        return;
      }
      self = this;
      best = Coder.pills.find('#pills-best');
      pillsFound = false;
      panel = Coder.pillsPanel = new PillsPanel(this);
      panel.offsetX = 5;
      panel.offsetY = 5;
      panel.width(350);
      panel.addClass('i-coder-pills');
      panel.help = 'kb://mapping.html';
      panel.on('destroy', function() {
        panel.content = null;
        panel._header.contents().detach();
        return panel._footer.contents().detach();
      });
      best["do"](function() {
        var group, j, l, len, len1, recursiveLookup, ref, ref1, ref2, ref3, variable, variables;
        this.contents().detach();
        recursiveLookup = function(variables, group, nested) {
          var j, len, pill, ref;
          ref = group.pills;
          for (j = 0, len = ref.length; j < len; j++) {
            pill = ref[j];
            variables.push(self.variables[pill.code]);
            if (pill.pills) {
              recursiveLookup(variables, pill, true);
            }
          }
          return null;
        };
        ref2 = (ref = (ref1 = self.config) != null ? ref1.pills : void 0) != null ? ref : [];
        for (j = 0, len = ref2.length; j < len; j++) {
          group = ref2[j];
          if (!(((ref3 = group.pills) != null ? ref3.length : void 0) > 0)) {
            continue;
          }
          this.h4().text(group.name + " ").append(sim.span('.i-square').text(group.id)).write(" - " + group.description);
          pillsFound = true;
          variables = [];
          recursiveLookup(variables, group);
          for (l = 0, len1 = variables.length; l < len1; l++) {
            variable = variables[l];
            this.div('.i-coder-pill-row', function() {
              var sample;
              this.attr('data-level', variable.data('level'));
              if (variable.data('nested')) {
                this.addClass("i-coder-pill-row-nested i-coder-level-" + (variable.data('level')));
                this.attr('data-nth', variable.data('nth'));
                if (variable.data('level') > 1 || 'collection' !== variable.data('parent-type')) {
                  this.addClass('out');
                }
              }
              if (variable.data('level') === 0 && 'collection' === variable.data('type')) {
                this.addClass('open');
              }
              sample = variable.data('sample');
              this.div().append(variable);
              if (variable.data('has-children')) {
                this.addClass('i-coder-pill-row-has-children');
                return this.on('click', function(event) {
                  var collapsing, elm, index, level, ref4;
                  if (event.target !== event.currentTarget) {
                    return;
                  }
                  this.toggleClass('open');
                  collapsing = !this.hasClass('open');
                  elm = this;
                  level = parseInt(this.data('level')) + 1;
                  index = 1;
                  while (elm) {
                    elm = elm.next(".i-coder-pill-row-nested");
                    if (!elm) {
                      break;
                    }
                    if (parseInt(elm.data('level')) < level) {
                      break;
                    }
                    if (collapsing) {
                      if (elm.hasClass('i-coder-pill-row-has-children')) {
                        elm.removeClass('open');
                      }
                      elm.addClass('out');
                    } else {
                      if (parseInt(elm.data('level')) === level && parseInt(elm.data('nth')) === index) {
                        elm.toggleClass('out');
                        index++;
                      }
                    }
                  }
                  return (ref4 = Coder.pillsPanel) != null ? ref4._resize() : void 0;
                });
              } else {
                if (sample != null) {
                  return this.append(sample);
                }
              }
            });
          }
        }
        return null;
      });
      panel.content = Coder.pills;
      if (self.config.showErasePill) {
        sim(".i-coder-pills .i-coder-keyword[data-name='erase']").show();
      } else {
        sim(".i-coder-pills .i-coder-keyword[data-name='erase']").hide();
      }
      panel._header.append(Coder.pillsHeader);
      panel._header.append(sim.button('.close', function() {
        this.attr('type', 'button');
        this.i('.far.fa-times');
        return this.on('click', function(event) {
          event.preventDefault();
          event.stopPropagation();
          self._hidePills();
          return false;
        });
      }));
      panel._header.append(sim.button('.help', function() {
        this.attr('type', 'button');
        this.i('.fas.fa-question');
        return this.on('click', function(event) {
          event.preventDefault();
          event.stopPropagation();
          sim(window).openHelp(panel.help);
          return false;
        });
      }));
      if (!pillsFound) {
        Coder.pillsHeader.find('a[href="#pills-best"]').hide();
        return Coder.pillsHeader.find('a[href="#pills-general"]').tab('show');
      } else {
        return Coder.pillsHeader.find('a[href="#pills-best"]').show().tab('show');
      }
    };

    Coder.prototype._hidePills = function() {
      if (!Coder.pillsPanel) {
        return;
      }
      Coder.pillsPanel.content = null;
      Coder.pillsPanel.close();
      return Coder.pillsPanel = null;
    };

    Coder.prototype._showHint = function(relative, pill) {
      var self;
      self = this;
      undelay(Coder._showHintTimer);
      return Coder._showHintTimer = delay(750, function() {
        var panel;
        if (Coder.hintPanel) {
          undelay(Coder.hintPanel._destroyTimer);
          if (Coder.hintPanel._relative === relative) {
            return;
          }
          Coder.hintPanel.destroy();
        }
        panel = Coder.hintPanel = new HintPanel(relative);
        panel.addClass('i-coder-hint');
        panel.offsetX = 6;
        panel.offsetY = 15;
        panel.width('auto');
        panel.position = relative.closest('#pills-best') ? 'left' : 'top';
        panel.content = sim.div(function() {
          var arrayIndexes, arrayLabels, example, j, len, ref, writeChanges;
          if (pill.theme) {
            panel.addClass("theme-" + (pill.theme.toHex().substr(1)));
            panel.find('.i-arrow-right').css('borderRightColor', pill.theme.toHex());
            panel.find('.i-arrow-left').css('borderLeftColor', pill.theme.toHex());
            panel.find('.i-arrow-top').css('borderTopColor', pill.theme.toHex());
            panel.find('.i-arrow-bottom').css('borderBottomColor', pill.theme.toHex());
          } else {
            if (relative.hasClass('i-coder-variable')) {
              panel.addClass('i-coder-hint-variable');
            }
            if (relative.hasClass('i-coder-operator')) {
              panel.addClass('i-coder-hint-operator');
            }
            if (relative.hasClass('i-coder-function')) {
              panel.addClass('i-coder-hint-function');
            }
            if (relative.hasClass('i-coder-keyword')) {
              panel.addClass('i-coder-hint-keyword');
            }
          }
          this.h6(function() {
            this.text(pill.name);
            if (pill.type) {
              return this.small().text(pill.type);
            }
          });
          if (pill.raw && pill.name !== pill.raw) {
            this.p('.i-raw').text(pill.raw);
          }
          if (pill.help) {
            this.p().text(pill.help);
          }
          if (pill.example) {
            ref = pill.example.trim().split(/\n/g);
            for (j = 0, len = ref.length; j < len; j++) {
              example = ref[j];
              example = example.match(/^([^\{]+)(?:\{([^\}]*)\})?$/);
              if (example) {
                this.pre(function() {
                  this.code(function() {
                    this.text(example[1]);
                    return this.visualizer({
                      type: 'iml',
                      imlCode: true
                    });
                  });
                  if (example[2]) {
                    return this.div().text("= " + example[2]);
                  }
                });
              }
            }
          }
          if (!relative.closest('.i-coder-pills')) {
            arrayIndexes = [];
            arrayLabels = [];
            relative.data('name').replace(/\[(\d*)\]/g, function(a, b) {
              arrayIndexes.push(b ? parseInt(b) : 1);
              return null;
            });
            writeChanges = function() {
              var i;
              i = 0;
              relative.data('name', pill.code.replace(/\[\d*\]/g, function() {
                return "[" + arrayIndexes[i++] + "]";
              }));
              i = 0;
              relative.data('prefix', pill.prefix.replace(/\[1\]/g, function(a, b) {
                return "[" + arrayIndexes[i++] + "]";
              }));
              if (pill.node && imt.config.showIDs) {
                relative.data('prefix', pill.node.id + ". " + (relative.data('prefix')));
              }
              if (arrayIndexes.length - 1 === i && /\[(\d*|n)\]/.test(pill.name)) {
                relative.text("[" + arrayIndexes[i] + "]");
              }
              return self._pull(self.value);
            };
            if (arrayIndexes.length) {
              arrayLabels = ("" + (pill.prefix || '') + pill.name).split(/\[(?:1|n)*\](?:\:\s|$)/g);
              return this.dl('.i-coder-arraycfg', function() {
                var fn, i, index, l, len1;
                fn = (function(_this) {
                  return function(i) {
                    return _this.dl(function() {
                      this.dt().text(arrayLabels[i] || '');
                      return this.dd().input('.form-control.input-sm', function() {
                        this.attr('type', 'text');
                        this.val(index);
                        return this.on('input blur', function() {
                          arrayIndexes[i] = parseInt(this.val()) || 1;
                          return writeChanges();
                        });
                      });
                    });
                  };
                })(this);
                for (i = l = 0, len1 = arrayIndexes.length; l < len1; i = ++l) {
                  index = arrayIndexes[i];
                  fn(i);
                }
                return null;
              });
            }
          }
        });
        panel.on('mouseenter', function() {
          return undelay(panel._destroyTimer);
        });
        return panel.on('mouseleave', function() {
          return panel._destroyTimer = delay(250, function() {
            if (Coder.hintPanel === panel) {
              Coder.hintPanel = null;
            }
            return panel.destroy();
          });
        });
      });
    };

    Coder.prototype._hideHint = function() {
      var ref;
      undelay(Coder._showHintTimer);
      return (ref = Coder.hintPanel) != null ? ref._destroyTimer = delay(250, function() {
        var ref1;
        if ((ref1 = Coder.hintPanel) != null) {
          ref1.destroy();
        }
        return Coder.hintPanel = null;
      }) : void 0;
    };

    Coder.prototype._normalize = function() {
      var anchor, child, container, contents, crap, err, index, j, last, len, next, node, offset, ref, ref1, ref2, ref3, ref4, ref5, res, root, text;
      ref = GET_CARET(), anchor = ref.anchor, offset = ref.offset;
      crap = null;
      node = null;
      root = this._root.__dom;
      last = null;
      child = null;
      index = null;
      if (root.childNodes.length) {
        next = root.firstChild;
        while (next) {
          child = next;
          if (child.nodeType === 1) {
            if (child.nodeName.toUpperCase() === 'SPAN' && child.className.indexOf('i-coder-pill') !== -1) {
              crap = (function() {
                var j, len, ref1, results;
                ref1 = child.childNodes;
                results = [];
                for (j = 0, len = ref1.length; j < len; j++) {
                  node = ref1[j];
                  if (node.nodeType !== 3) {
                    results.push(node);
                  }
                }
                return results;
              })();
              if (crap.length) {
                text = child.textContent;
                while (child.firstChild) {
                  child.removeChild(child.firstChild);
                }
                child.appendChild(document.createTextNode(text));
              }
            } else if (child.nodeName.toUpperCase() === 'SPAN' && child.className.indexOf('i-coder-newline') !== -1) {
              if (child.textContent !== '\n') {
                while (child.firstChild) {
                  child.removeChild(child.firstChild);
                }
                child.appendChild(document.createTextNode('\n'));
              }
            } else {
              if (child.childNodes.length) {
                next = child.firstChild;
                while (child.firstChild) {
                  root.insertBefore(child.firstChild, child);
                }
                root.removeChild(child);
                continue;
              } else {
                next = child.nextSibling;
                root.removeChild(child);
                continue;
              }
            }
          } else if ((last != null ? last.nodeType : void 0) === 3 && child.nodeType === 3 && child.nodeValue.length > 0) {
            if (IS_SPACER(child)) {
              if (child === anchor) {
                anchor = last;
                offset = last.length;
              }
              next = child.nextSibling;
              root.removeChild(child);
              continue;
            }
            if (child === anchor) {
              anchor = last;
              offset = last.length + offset;
            }
            last.nodeValue += child.nodeValue;
            next = child.nextSibling;
            root.removeChild(child);
            continue;
          } else if (child.nodeType === 3 && child.nodeValue.length === 0) {
            next = child.nextSibling;
            root.removeChild(child);
            continue;
          }
          last = child;
          next = child.nextSibling;
        }
      }
      if (root.childNodes.length) {
        next = root.firstChild;
        while (next) {
          child = next;
          if (child.nodeType === 3) {
            res = /{{([\s\S]*?)}}/g.exec(child.nodeValue);
            if (res) {
              container = sim.div();
              AST_TO_DOM.call(this, IML.parse(res[0]), container, this.variables);
              ref1 = IML.errors;
              for (j = 0, len = ref1.length; j < len; j++) {
                err = ref1[j];
                console.warn(err);
              }
              contents = container.contents();
              if (res.index + res[0].length === child.nodeValue.length) {
                child.nodeValue = child.nodeValue.substr(0, res.index);
                if (anchor === child && offset > res.index) {
                  if (((ref2 = child.nextSibling) != null ? ref2.nodeType : void 0) !== 3) {
                    text = document.createTextNode("\u200B");
                    if (child.nextSibling != null) {
                      root.insertBefore(text, child.nextSibling);
                    } else {
                      root.appendChild(text);
                    }
                    anchor = text;
                    offset = 1;
                  } else {
                    anchor = child.nextSibling;
                    offset = IS_SPACER(child) ? 1 : 0;
                  }
                }
                if (child.nextSibling != null) {
                  contents.insertBefore(child.nextSibling);
                } else {
                  contents.appendTo(root);
                }
                last = contents[contents.length - 1];
                next = last != null ? last.nextSibling : void 0;
                continue;
              } else if (res.index === 0) {
                if (anchor === child) {
                  if (offset < res[0].length) {
                    if (((ref3 = child.previousSibling) != null ? ref3.nodeType : void 0) !== 3) {
                      text = document.createTextNode("\u200B");
                      root.insertBefore(text, child);
                      anchor = text;
                      offset = 1;
                    } else {
                      anchor = child.previousSibling;
                      offset = anchor.nodeValue.length;
                    }
                  } else {
                    offset -= res[0].length;
                  }
                }
                child.nodeValue = child.nodeValue.substr(res[0].length);
                contents.insertBefore(child);
                next = child;
                last = child.previousSibling;
                continue;
              } else {
                next = child.splitText(res.index);
                next.nodeValue = next.nodeValue.substr(res[0].length);
                contents.insertBefore(next);
                if (anchor === child && offset > res.index) {
                  anchor = next;
                  offset = Math.max(0, offset - (res.index + res[0].length));
                }
                last = next.previousSibling;
                continue;
              }
            }
          }
          last = child;
          next = child.nextSibling;
        }
      }
      if (root.childNodes.length) {
        next = root.firstChild;
        while (next) {
          child = next;
          if (child.nodeType === 3) {
            index = child.nodeValue.indexOf("\u200B");
            if (index !== -1) {
              if (child.nodeValue.length > 1) {
                child.nodeValue = child.nodeValue.replace(/\u200B/g, "");
                if (child === anchor) {
                  offset = index < offset ? offset - 1 : offset;
                }
              } else {
                if (((child.previousSibling != null) && child.previousSibling.nodeType !== 1) || ((child.nextSibling != null) && child.nextSibling.nodeType !== 1)) {
                  next = child.nextSibling;
                  root.removeChild(child);
                  continue;
                }
              }
            }
          } else if (child.nodeType === 1 && (last != null ? last.nodeType : void 0) === 1) {
            last = document.createTextNode("\u200B");
            root.insertBefore(last, child);
          }
          last = child;
          next = child.nextSibling;
        }
      }
      if (((ref4 = root.firstChild) != null ? ref4.nodeType : void 0) === 1) {
        text = document.createTextNode("\u200B");
        root.insertBefore(text, root.firstChild);
      }
      if (((ref5 = root.lastChild) != null ? ref5.nodeType : void 0) === 1) {
        text = document.createTextNode("\u200B");
        root.appendChild(text);
      }
      if (this._root.is(':focus')) {
        if (anchor != null) {
          SET_CARET(anchor, offset);
        } else {
          if (root.lastChild != null) {
            SET_CARET(root.lastChild, root.lastChild.nodeValue.length);
          }
        }
      }
      return null;
    };

    Coder.prototype._pull = function(value) {
      this._input.val(value);
      return this._input.emit('change');
    };

    Coder.prototype.destroy = function() {
      this._hidePills();
      return this.remove();
    };

    Coder.prototype.detach = function() {
      Coder.__super__.detach.call(this);
      return this._hidePills();
    };

    Coder.prototype.focus = function() {
      return this._root.focus();
    };

    Coder.prototype.insert = function(value) {
      var anchor, offset, ref;
      if ((value == null) || value === '') {
        return;
      }
      if (!this._root.__dom.firstChild) {
        this._root.append(value);
        return this.emit('input');
      }
      ref = GET_CARET(), anchor = ref.anchor, offset = ref.offset;
      if (!anchor) {
        return;
      }
      INJECT_TEXT_NODE(anchor, offset, value);
      return this.emit('input');
    };

    return Coder;

  })(SIMElement);

  PillsPanel = (function(superClass) {
    extend(PillsPanel, superClass);

    function PillsPanel() {
      return PillsPanel.__super__.constructor.apply(this, arguments);
    }

    PillsPanel.property('position', {
      get: function() {
        if (sim(window).width() < 700) {
          return 'top';
        } else {
          return this.__position;
        }
      },
      set: function(value) {
        if (this.__position === value) {
          return;
        }
        this.__position = value;
        return this._move();
      }
    });

    return PillsPanel;

  })(Panel);

  HintPanel = (function(superClass) {
    extend(HintPanel, superClass);

    function HintPanel() {
      return HintPanel.__super__.constructor.apply(this, arguments);
    }

    return HintPanel;

  })(Panel);

}).call(this);
