import React, {
  createContext,
  useCallback,
  useContext,
  useEffect,
  useState,
} from "react";
import APP_CONFIG from "../config/config";
import { generateCodeChallenge, generateCodeVerifier } from "../utils/auth";

// Create the Adobe IMSAuthContext to provide the authentication state to the children components
const IMSAuthContext = createContext(null);

const STORAGE_KEYS = {
  ACCESS_TOKEN: "adobe_ims_access_token",
  CODE_VERIFIER: "code_verifier",
};

// Define the code challenge methods, S256 is the recommended method for production and plain is the recommended method for debugging
const AUTH_METHODS = {
  S256: "S256",
  PLAIN: "plain",
};

// Create the Adobe IMSAuthProvider component to provide the authentication state to the children components
export const IMSAuthProvider = ({ children }) => {
  const [authState, setAuthState] = useState({
    isLoggedIn: false,
    accessToken: null,
    isProcessingCallback: false,
  });

  // Helper functions
  const updateAuthState = useCallback((updates) => {
    setAuthState((prev) => ({ ...prev, ...updates }));
  }, []);

  const handleStorageToken = useCallback(
    (token) => {
      if (token) {
        localStorage.setItem(STORAGE_KEYS.ACCESS_TOKEN, token);
        updateAuthState({ isLoggedIn: true, accessToken: token });
      }
    },
    [updateAuthState]
  );

  const clearAuthData = useCallback(() => {
    localStorage.removeItem(STORAGE_KEYS.ACCESS_TOKEN);
    localStorage.removeItem(STORAGE_KEYS.CODE_VERIFIER);
    updateAuthState({ isLoggedIn: false, accessToken: null });
  }, [updateAuthState]);

  // Initialize auth state from storage
  useEffect(() => {
    const storedToken = localStorage.getItem(STORAGE_KEYS.ACCESS_TOKEN);
    handleStorageToken(storedToken);
  }, [handleStorageToken]);

  // Get authorization URL parameters
  const getAuthParams = useCallback((method, codeChallenge, codeVerifier) => {
    const baseParams = {
      client_id: APP_CONFIG.adobe.adc.clientId,
      scope: APP_CONFIG.adobe.adc.scopes,
      response_type: "code",
      redirect_uri: APP_CONFIG.adobe.spa.redirectUri,
      code_challenge_method: method,
    };

    return {
      ...baseParams,
      code_challenge:
        method === AUTH_METHODS.S256 ? codeChallenge : codeVerifier,
    };
  }, []);

  // Login function
  const login = async () => {
    try {
      const codeVerifier = generateCodeVerifier();
      const codeChallenge = generateCodeChallenge(codeVerifier);

      localStorage.setItem(STORAGE_KEYS.CODE_VERIFIER, codeVerifier);

      const params = new URLSearchParams(
        getAuthParams(AUTH_METHODS.S256, codeChallenge, codeVerifier)
      );

      window.location.href = `${
        APP_CONFIG.adobe.ims.authorizationEndpoint
      }?${params.toString()}`;
    } catch (error) {
      console.error("Login initialization failed:", error);
      throw error;
    }
  };

  // Exchange the authorization code for an access token
  const exchangeCodeForToken = useCallback(async (code) => {
    const codeVerifier = localStorage.getItem(STORAGE_KEYS.CODE_VERIFIER);

    if (!codeVerifier) {
      throw new Error("No code verifier found");
    }

    const response = await fetch(APP_CONFIG.adobe.ims.tokenEndpoint, {
      method: "POST",
      headers: { "Content-Type": "application/x-www-form-urlencoded" },
      body: new URLSearchParams({
        grant_type: "authorization_code",
        client_id: APP_CONFIG.adobe.adc.clientId,
        code_verifier: codeVerifier,
        code,
        redirect_uri: `${window.location.origin}/callback`,
      }),
    });

    if (!response.ok) {
      throw new Error("Token request failed");
    }

    return response.json();
  }, []);

  // Handle the callback from the Adobe IMS authorization endpoint
  const handleCallback = async (code) => {
    if (authState.isProcessingCallback) return;

    try {
      updateAuthState({ isProcessingCallback: true });

      const data = await exchangeCodeForToken(code);

      if (data.access_token) {
        handleStorageToken(data.access_token);
        localStorage.removeItem(STORAGE_KEYS.CODE_VERIFIER);
      }
    } catch (error) {
      console.error("Error exchanging code for token:", error);
      throw error;
    } finally {
      updateAuthState({ isProcessingCallback: false });
    }
  };

  // Provide the AuthContext to the children components
  const contextValue = {
    isLoggedIn: authState.isLoggedIn,
    accessToken: authState.accessToken,
    login,
    logout: clearAuthData,
    handleCallback,
  };

  return (
    <IMSAuthContext.Provider value={contextValue}>
      {children}
    </IMSAuthContext.Provider>
  );
};

// Custom hook to use the AuthContext
export const useIMSAuth = () => {
  const context = useContext(IMSAuthContext);
  if (!context) {
    throw new Error("useIMSAuth must be used within an IMSAuthProvider");
  }
  return context;
};
