import React, { useCallback } from "react";
import { useNavigate } from "react-router-dom";
import { useIMSAuth } from "../context/IMSAuthContext";
import AemApiEndpointDisplay from "./AemApiEndpointDisplay";
import APP_CONFIG from "../config/config";

// Constants
const ROUTES = {
  APIS: "/invoke-aem-apis",
};

const BUTTON_TEXT = {
  LOGIN: "Adobe IMS Login",
  CONTINUE: "Continue to APIs",
  REVIEW_CONFIG: "Please Review Configuration",
};

function Home() {
  const { isLoggedIn, login } = useIMSAuth();
  const navigate = useNavigate();

  // Check if required configuration is present
  const isConfigValid = useCallback(() => {
    //make sure APP_CONFIG is defined
    if (
      !APP_CONFIG ||
      !APP_CONFIG.adobe ||
      !APP_CONFIG.adobe.aem ||
      !APP_CONFIG.adobe.adc
    ) {
      return false;
    }

    const requiredConfigs = [
      APP_CONFIG.adobe.aem.hostname,
      APP_CONFIG.adobe.adc.clientId,
      APP_CONFIG.adobe.adc.scopes,
      APP_CONFIG.adobe.ims.authorizationEndpoint,
      APP_CONFIG.adobe.ims.tokenEndpoint,
    ];
    return requiredConfigs.every((config) => config && config.length > 0);
  }, []);

  // Handlers
  const handleButtonClick = useCallback(() => {
    if (!isConfigValid()) return;
    isLoggedIn ? navigate(ROUTES.APIS) : login();
  }, [isLoggedIn, login, navigate, isConfigValid]);

  // UI Components
  const renderConfigWarning = () => {
    if (isConfigValid()) return null;

    return (
      <div className="config-warning">
        <h3>⚠️ Configuration Required</h3>
        <p>Please set up the following in your <code>.env</code> file:</p>
        <ul>
          <li>REACT_APP_AEM_HOST</li>
          <li>REACT_APP_IMS_CLIENT_ID</li>
          <li>REACT_APP_IMS_SCOPE</li>
        </ul>
        <p>See the README for more details.</p>
      </div>
    );
  };

  const renderHeader = () => (
    <div className="header-text">
      <h1>WKND SPA & AEM APIs - Demo App</h1>
      <p>
        A demo WKND Single Page Application (SPA) that uses OpenAPI-based AEM
        APIs from the <em>client-side</em>, authenticating on behalf of the user
        via OAuth 2.0 PKCE (Single Page App flow).
      </p>
      <AemApiEndpointDisplay />
      {renderConfigWarning()}
    </div>
  );

  const renderActionButton = () => (
    <button
      id="try-now-btn"
      className="btn"
      onClick={handleButtonClick}
      disabled={!isConfigValid()}
      style={!isConfigValid() ? { backgroundColor: "#6c757d" } : {}}
    >
      {!isConfigValid()
        ? BUTTON_TEXT.REVIEW_CONFIG
        : isLoggedIn
        ? BUTTON_TEXT.CONTINUE
        : BUTTON_TEXT.LOGIN}
    </button>
  );

  return (
    <>
      {renderHeader()}
      <main className="intro-main">{renderActionButton()}</main>
    </>
  );
}

export default Home;
