import React, { useState, useCallback } from "react";
import APP_CONFIG from "../config/config";
import AemApiEndpointDisplay from "./AemApiEndpointDisplay";

// Constants
const TABS = {
  SITES: "sites-api",
  ASSETS: "assets-folder-api",
};

const API_PATHS = {
  FOLDERS: "adobe/folders/",
  CF_MODELS: "adobe/sites/cf/models",
};

const BASE_DAM_PATH = "/content/dam/wknd-shared/en";

function InvokeAemApis() {
  // State management
  const [state, setState] = useState({
    activeTab: TABS.SITES,
    cfModels: null,
    damFolders: null,
    createdDamFolder: null,
    isLoading: false,
    error: null,
    deletingFolderId: null,
  });

  // Helper function to update state
  const updateState = useCallback((updates) => {
    setState((prev) => ({ ...prev, ...updates }));
  }, []);

  // Common API request helper
  const makeApiRequest = useCallback(
    async ({ endpoint, method = "GET", passAPIKey = false, body = null }) => {
      const token = localStorage.getItem("adobe_ims_access_token");
      if (!token) {
        throw new Error("No access token available. Please login again.");
      }

      const headers = {
        Authorization: `Bearer ${token}`,
        "Content-Type": "application/json",
        ...(passAPIKey && { "x-api-key": APP_CONFIG.adobe.adc.clientId }),
      };

      const response = await fetch(
        `${APP_CONFIG.adobe.aem.hostname}${endpoint}`,
        {
          method,
          headers,
          ...(body && { body: JSON.stringify(body) }),
        }
      );

      if (!response.ok) {
        throw new Error(`API request failed: ${response.statusText}`);
      }

      return method === "DELETE" ? null : response.json();
    },
    []
  );

  // API Functions
  const fetchContentFragmentModels = useCallback(async () => {
    try {
      updateState({ isLoading: true, error: null });
      const data = await makeApiRequest({
        endpoint: `${API_PATHS.CF_MODELS}?cursor=0&limit=10&projection=summary`,
      });
      updateState({ cfModels: data.items });
    } catch (err) {
      updateState({ error: err.message });
      console.error("Error fetching CF models:", err);
    } finally {
      updateState({ isLoading: false });
    }
  }, [makeApiRequest, updateState]);

  const fetchDamFolders = useCallback(async () => {
    try {
      updateState({ isLoading: true, error: null, createdDamFolder: null });
      const data = await makeApiRequest({
        endpoint: `${API_PATHS.FOLDERS}?path=${encodeURIComponent(
          BASE_DAM_PATH
        )}&limit=10`,
        passAPIKey: true,
      });
      updateState({ damFolders: data });
    } catch (err) {
      updateState({ error: err.message });
      console.error("Error fetching DAM folders:", err);
    } finally {
      updateState({ isLoading: false });
    }
  }, [makeApiRequest, updateState]);

  const createDamFolder = useCallback(
    async (event) => {
      event.preventDefault();
      const folderName = new FormData(event.target).get("folderName");

      if (!folderName) {
        updateState({ error: "Folder name is required" });
        return;
      }

      try {
        updateState({ isLoading: true, error: null });
        await makeApiRequest({
          endpoint: API_PATHS.FOLDERS,
          method: "POST",
          passAPIKey: true,
          body: [
            {
              title: folderName,
              path: `${BASE_DAM_PATH}/${folderName}`,
            },
          ],
        });
        await fetchDamFolders();
      } catch (err) {
        updateState({ error: err.message });
        console.error("Error creating DAM folder:", err);
      } finally {
        updateState({ isLoading: false });
      }
    },
    [makeApiRequest, fetchDamFolders, updateState]
  );

  const handleDeleteFolder = useCallback(
    async (folderId) => {
      try {
        updateState({ deletingFolderId: folderId });
        const queryParams = new URLSearchParams({
          recursive: "false",
          force: "false",
        });

        await makeApiRequest({
          endpoint: `${API_PATHS.FOLDERS}${folderId}?${queryParams.toString()}`,
          method: "DELETE",
          passAPIKey: true,
        });
        await fetchDamFolders();
      } catch (err) {
        updateState({ error: err.message });
        console.error("Error deleting folder:", err);
      } finally {
        updateState({ deletingFolderId: null });
      }
    },
    [makeApiRequest, fetchDamFolders, updateState]
  );

  // UI Components
  const renderHeader = () => (
    <div className="header-text">
      <h1>WKND SPA & AEM APIs - Demo App</h1>
      <p>
        Use OpenAPI-based AEM APIs to read and update data owned by the
        logged-in user from the <em>client-side</em>, authenticated via OAuth
        2.0 PKCE (Single Page App flow).
      </p>
      <AemApiEndpointDisplay />
    </div>
  );

  const renderTabs = () => (
    <div className="tabs">
      <button
        className={`tab-button ${
          state.activeTab === TABS.SITES ? "active" : ""
        }`}
        onClick={() => updateState({ activeTab: TABS.SITES })}
      >
        Sites API
      </button>
      <button
        className={`tab-button ${
          state.activeTab === TABS.ASSETS ? "active" : ""
        }`}
        onClick={() => updateState({ activeTab: TABS.ASSETS })}
      >
        Assets - Folders API
      </button>
    </div>
  );

  const renderLoadingAndError = () => (
    <>
      {state.isLoading && (
        <div className="loading-container">
          <div className="spinner"></div>
        </div>
      )}
      {state.error && <div className="error-message">Error: {state.error}</div>}
    </>
  );

  const renderFoldersTable = () =>
    state.damFolders && (
      <div className="table-container">
        <table className="folders-table">
          <thead>
            <tr>
              <th>Title</th>
              <th>Name</th>
              <th>Path</th>
              <th>Folder ID</th>
              <th>Actions</th>
            </tr>
          </thead>
          <tbody>
            <tr className="parent-row">
              <td>{state.damFolders.self.title}</td>
              <td>{state.damFolders.self.name}</td>
              <td>{state.damFolders.self.path}</td>
              <td>{state.damFolders.self.folderId}</td>
              <td>-</td>
            </tr>
            {state.damFolders.children.map((folder) => (
              <tr key={folder.folderId}>
                <td>{folder.title}</td>
                <td>{folder.name}</td>
                <td>{folder.path}</td>
                <td>{folder.folderId}</td>
                <td>
                  <button
                    className="delete-button"
                    onClick={() => handleDeleteFolder(folder.folderId)}
                    disabled={state.deletingFolderId !== null}
                  >
                    {state.deletingFolderId === folder.folderId ? (
                      <div className="spinner-small"></div>
                    ) : (
                      "Delete"
                    )}
                  </button>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    );

  return (
    <>
      {renderHeader()}
      <main className="intro-main">
        {renderTabs()}
        <div className="tab-content active">
          {state.activeTab === TABS.SITES ? (
            <div className="section">
              <h2 className="section-title">Content Fragment Models</h2>
              <div className="section-description">
                <p>
                  For demo purposes, we are fetching the first 10 Content
                  Fragment Models.
                </p>
              </div>
              <div className="actions-container">
                <div className="fetch-section">
                  <button
                    className="secondary-button"
                    onClick={fetchContentFragmentModels}
                  >
                    Fetch Content Fragment Models
                  </button>
                </div>
              </div>
              {renderLoadingAndError()}
              {/* Content Fragment Models Table */}
              {state.cfModels && (
                <div className="table-container">
                  <table className="folders-table">
                    <thead>
                      <tr>
                        <th>Name</th>
                        <th>Path</th>
                        <th>Status</th>
                        <th>Created</th>
                        <th>Created By</th>
                      </tr>
                    </thead>
                    <tbody>
                      {state.cfModels.map((model) => (
                        <tr key={model.id}>
                          <td>{model.name}</td>
                          <td>{model.path}</td>
                          <td>{model.status}</td>
                          <td>
                            {new Date(model.created.at).toLocaleDateString()}
                          </td>
                          <td>{model.created.fullName || model.created.by}</td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              )}
            </div>
          ) : (
            <div className="section">
              <h2 className="section-title">DAM Folders</h2>
              <div className="section-description">
                <p>For demo purposes, we have two options:</p>
                <ul className="options-list">
                  <li>Fetch first 10 DAM Folders</li>
                  <li>Create or delete a DAM Folder</li>
                </ul>
                <p>
                  For the sake of simplicity, we are using the{" "}
                  <em>{BASE_DAM_PATH}</em> path for both the options.
                </p>
              </div>
              <div className="actions-container">
                <div className="fetch-section">
                  <button
                    className="secondary-button"
                    onClick={fetchDamFolders}
                  >
                    Fetch DAM Folders
                  </button>
                </div>
                <div className="create-section">
                  <form
                    onSubmit={createDamFolder}
                    className="create-folder-form"
                  >
                    <input
                      type="text"
                      name="folderName"
                      placeholder="Folder Name"
                      className="folder-input"
                      required
                    />
                    <button
                      type="submit"
                      className="primary-button"
                      disabled={state.isLoading}
                    >
                      Create DAM Folder
                    </button>
                  </form>
                </div>
                {renderLoadingAndError()}
              </div>
              {renderFoldersTable()}
            </div>
          )}
        </div>
      </main>
    </>
  );
}

export default InvokeAemApis;
