import React, { useEffect, useRef, memo } from 'react';
import { useNavigate, useLocation } from 'react-router-dom';
import { useIMSAuth } from '../context/IMSAuthContext';

// Constants
const ROUTES = {
  HOME: '/',
  APIS: '/invoke-aem-apis'
};

const MESSAGES = {
  LOGGING_IN: 'Logging you in...',
  ERROR: 'Error during callback:'
};

// Loading component
const LoadingMessage = memo(() => (
  <div className="loading-container">
    <div className="spinner"></div>
    <div>{MESSAGES.LOGGING_IN}</div>
  </div>
));

LoadingMessage.displayName = 'LoadingMessage';

function Callback() {
  const { handleCallback } = useIMSAuth();
  const navigate = useNavigate();
  const location = useLocation();
  const hasProcessedCode = useRef(false);

  useEffect(() => {
    const processAuthCode = async (code) => {
      try {
        await handleCallback(code);
        navigate(ROUTES.APIS);
      } catch (error) {
        console.error(MESSAGES.ERROR, error);
        navigate(ROUTES.HOME);
      }
    };

    const handleAuthCallback = () => {
      if (hasProcessedCode.current) return;

      const searchParams = new URLSearchParams(location.search);
      const authCode = searchParams.get('code');

      if (authCode) {
        hasProcessedCode.current = true;
        processAuthCode(authCode);
      } else {
        // If no auth code is present, redirect to home
        navigate(ROUTES.HOME);
      }
    };

    handleAuthCallback();

    // Cleanup function
    return () => {
      hasProcessedCode.current = false;
    };
  }, [handleCallback, navigate, location]);

  return <LoadingMessage />;
}

// Add display name for debugging
Callback.displayName = 'Callback';

// Memoize the component to prevent unnecessary re-renders
export default memo(Callback); 