import express from "express";
import URLSearchParams from "url-search-params";
import fetch from "node-fetch";
import config from "../config.js";

const router = express.Router();

// Extracting Adobe IMS and Developer Console configurations
const { ims: adobeIMSConfig, adc: adobeADCConfig } = config.adobe;

// Dynamic redirect URI based on environment
const redirectUri =
  process.env.REDIRECT_URI || `https://localhost:3001/callback`;

// Route to initiate Adobe IMS user authentication
router.get("/auth", (req, res) => {
  // Redirect user to Adobe IMS authorization endpoint
  try {
    // Constructing the authorization URL
    const params = new URLSearchParams({
      client_id: adobeADCConfig.clientId,
      redirect_uri: redirectUri,
      response_type: "code",
    });

    // Append scopes if defined in configuration
    if (adobeADCConfig?.scopes) params.append("scope", adobeADCConfig.scopes);

    // Redirect user to Adobe IMS authorization URL
    const imsAuthorizationUrl = `${
      adobeIMSConfig.authorizationEndpoint
    }?${params.toString()}`;

    res.redirect(imsAuthorizationUrl);
  } catch (error) {
    console.error("Error initiating Adobe IMS authentication:", error);
    res.status(500).send("Unable to initiate authentication");
  }
});

// Callback route to exchange authorization code for access token
router.get("/callback", async (req, res) => {
  // Extracting authorization code from the query parameters
  const authorizationCode = req.query.code;

  if (!authorizationCode) {
    return res.status(400).send("Missing authorization code");
  }

  // Exchange authorization code for access token
  try {
    // Fetch access token from Adobe IMS token endpoint
    const response = await fetch(adobeIMSConfig.tokenEndpoint, {
      method: "POST",
      headers: {
        "Content-Type": "application/x-www-form-urlencoded",
        Authorization: `Basic ${Buffer.from(
          `${adobeADCConfig.clientId}:${adobeADCConfig.clientSecret}`
        ).toString("base64")}`,
      },
      body: new URLSearchParams({
        code: authorizationCode,
        grant_type: "authorization_code",
      }),
    });

    if (!response.ok) {
      console.error("Failed to fetch access token:", response.statusText);
      return res.status(500).send("Failed to fetch access token");
    }

    const data = await response.json();

    if (!data.access_token) {
      console.error("Access token missing in the response:", data);
      return res.status(500).send("Invalid response from token endpoint");
    }

    // For debugging purposes
    console.log("Access token:", data.access_token);

    // Store the access token in the session
    req.session.accessToken = data.access_token;

    // Redirect user to update product attributes
    res.redirect("/update-product-attributes");
  } catch (error) {
    console.error("Error exchanging authorization code:", error);
    res.status(500).send("Error during token exchange");
  }
});

export default router;
