document.addEventListener("DOMContentLoaded", () => {
  const tabButtons = document.querySelectorAll(".tab-button");
  const tabContents = document.querySelectorAll(".tab-content");
  const assetDropdown = document.getElementById("asset-id");
  const updateMetadataButton = document.getElementById("update-aem-btn");
  const loadingSpinner = document.getElementById("spinner");
  const errorDisplay = document.getElementById("error-message");

  // Utility functions
  const showElement = (element) => {
    element.style.display = "block";
  };
  const hideElement = (element) => {
    element.style.display = "none";
  };
  const setError = (message, color = "red") => {
    errorDisplay.textContent = message;
    errorDisplay.style.color = color;
    showElement(errorDisplay);
  };

  // Tab switching logic
  tabButtons.forEach((button) => {
    button.addEventListener("click", () => {
      tabButtons.forEach((btn) => btn.classList.remove("active"));
      tabContents.forEach((content) => content.classList.remove("active"));

      button.classList.add("active");
      const targetTabId = button.getAttribute("data-tab");
      document.getElementById(targetTabId).classList.add("active");
    });
  });

  // Fetch metadata on asset change
  assetDropdown.addEventListener("change", async () => {
    const selectedAssetId = assetDropdown.value;

    if (!selectedAssetId) {
      setError("Please select an asset.");
      return;
    }

    showElement(loadingSpinner);
    hideElement(errorDisplay);

    try {

      // Clear previous metadata
      document
        .querySelectorAll("#aemassets-info input, #aemassets-info textarea")
        .forEach((input) => {
          input.value = "";
        });

      const metadata = await fetchAEMAssetMetadata(selectedAssetId);

      if (metadata) {
        console.log("Fetched Metadata:", metadata);
        Object.entries(metadata).forEach(([key, value]) => {
          const inputElement = document.getElementById(key.replace(":", "-"));
          if (inputElement) inputElement.value = value;
        });
      } else {
        throw new Error("No metadata found for the selected asset.");
      }
    } catch (error) {
      console.error("Error fetching metadata:", error);
      setError(`Error fetching metadata: ${error.message}`);
    } finally {
      hideElement(loadingSpinner);
    }
  });

  // Update metadata button click handler
  updateMetadataButton.addEventListener("click", async () => {
    const selectedAssetId = assetDropdown.value;

    if (!selectedAssetId) {
      setError("Please select an asset.");
      return;
    }

    const metadata = {};

    document
      .querySelectorAll("#aemassets-info input, #aemassets-info textarea")
      .forEach((input) => {
        metadata[input.id.replace("-", ":")] = input.value.trim();
      });

    if (!metadata["dc:title"] || !metadata["dc:description"]) {
      setError("Both Title and Description are required.");
      return;
    }

    showElement(loadingSpinner);
    hideElement(errorDisplay);

    try {
      const result = await updateAEMAssetMetadata(selectedAssetId, metadata);
      console.log("Update Metadata Response:", result);

      setError("Metadata updated successfully!", "green");
    } catch (error) {
      console.error("Error updating metadata:", error);
      setError(`Error updating metadata: ${error.message}`);
    } finally {
      hideElement(loadingSpinner);
    }
  });

  // Fetch metadata function
  async function fetchAEMAssetMetadata(assetId) {
    if (!navigator.onLine) {
      throw new Error(
        "You are offline. Please check your internet connection."
      );
    }

    const response = await fetch(`/api/getAEMAssetMetadata?assetId=${assetId}`);
    if (!response.ok) {
      switch (response.status) {
        case 401:
          throw new Error("Session expired. Please log in again.");
        case 400:
          throw new Error("Invalid asset ID provided.");
        default:
          throw new Error(`Failed to fetch metadata: ${response.status}`);
      }

      /*
      throw new Error(
        `Failed to fetch metadata. HTTP Status: ${response.status}`
      );
      */
    }

    return response.json();
  }

  // Update metadata function
  async function updateAEMAssetMetadata(assetId, metadata) {
    const response = await fetch("/api/updateAEMAssetMetadata", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ assetId, metadata }),
    });

    if (!response.ok) {
      switch (response.status) {
        case 401:
          throw new Error("Session expired. Please log in again.");
        case 400:
          throw new Error("Invalid asset ID provided.");
        default:
          throw new Error(`Unexpected error occurred: ${response.status}`);
      }
    }

    return response.json();
  }
});
