# REST APIs for Testing

This repository contains REST APIs and are used to make HTTPS calls from AEM for testing purposes. It contains the following APIs.

- Self-Signed Certificate API via `server.js`.
- Mutual TLS (mTLS) authentication API via `server-mtls.js`.

## How to run

To make the API available locally on HTTPS, run the following command

```shell
# To host the Self-Signed Certificate API
$ npm start

# To host the mTLS protected API
$ npm run start-mtls
```

## How to test

### Self-Signed Certificate API

The API can be accessed by visiting [https://127.0.0.1:3000/now](https://127.0.0.1:3000/now), however you have to accept the **Your connection is not private** warning.

It should return the current Date and Time as a JSON response.

`{"date":"2023-8-25","time":"13:46:42"}`

### Mutual TLS (mTLS) authentication API

Test the mTLS setup with cURL. Pay special attention to the `--cert` and `--key` flags. These flags are used to specify the client certificate and key respectively.

```shell
curl --cert client-cert.pem --key client-key.pem  -k https://127.0.0.1:3000/now
```

## Certificates

- The certificates used in this project are self-signed and generated using OpenSSL.
- The client and server certificates are signed by an internal Certification Authority (CA) certificate. Thus we first need to generate the internal Certification Authority (CA) certificate.
- The certificates are stored in the `certs` folder.
- The certificates are valid for 9999 days.
- The password for all the cert keys is `admin`.

## Cert Generation Command

Following are the commands used to generate the certificates.

### Internal Certification Authority (CA)

First we need to generate the internal Certification Authority (CA) certificate.

```shell
# Create an internal Certification Authority (CA) certificate
openssl req -new -x509 -days 9999 -keyout internal-ca-key.pem -out internal-ca-cert.pem
```

### Server Certificate

Next we need to generate the server certificate. This certificate is used by the server to authenticate itself to the client. In this case, its used inside the `server.js` and `server-mtls.js` files.

```shell
# Generate Key
openssl genrsa -out server-key.pem

# Generate CSR
openssl req -new -key server-key.pem -out server-csr.pem

# Generate certificate and sign with internal Certification Authority (CA)
openssl x509 -req -days 9999 -in server-csr.pem -CA internal-ca-cert.pem -CAkey internal-ca-key.pem -CAcreateserial -out server-cert.pem

# Verify certificate
openssl verify -CAfile internal-ca-cert.pem server-cert.pem
```

### Client Certificate

Finally we need to generate the client certificate. This certificate is used by the client to authenticate itself to the server, however only necessary in mutual TLS (mTLS) authentication setup. 

```shell
# Generate Key
openssl genrsa -out client-key.pem

# Generate CSR
openssl req -new -key client-key.pem -out client-csr.pem

# Generate certificate and sign with internal Certification Authority (CA)
openssl x509 -req -days 9999 -in client-csr.pem -CA internal-ca-cert.pem -CAkey internal-ca-key.pem -CAcreateserial -out client-cert.pem

# Verify certificate
openssl verify -CAfile internal-ca-cert.pem client-cert.pem
```



### Convert Private Key to DER format

```shell
openssl pkcs8 -topk8 -inform PEM -outform DER -in client-key.pem -out client-key.der -nocrypt
```


 